﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace MicroRWD.UI.MFIC
{
    public partial class ReadMifareCardMemory : Form
    {
        #region Private Properties

        Mifare_MemoryPanel parent;
        private const byte C_LAST_BLOCK = 0x3F;
        private const byte C_LAST_KEY = 0x1F;
        private bool updateStartFields = true;
        private bool updateEndFields = true;
        private bool updateKeyFields = true;
        private byte startAddress = 0x00;        
        private byte endAddress = C_LAST_BLOCK;
        private byte keyNumber = 0x00;
        private bool keyType = false;
        
        #endregion

        #region Constructors

        public ReadMifareCardMemory(Mifare_MemoryPanel _parent)
        {
            parent = _parent;
            InitializeComponent();
        }

        #endregion

        #region Private Methods

        private void refreshView()
        {
            if (updateStartFields)
            {
                // Save the current text selection
                int selectionStart = startAddrHexTextBox.SelectionStart;
                int selctionLength = startAddrHexTextBox.SelectionLength;

                // Update values
                this.startAddrHexTextBox.Text = startAddress.ToString("X2");
                this.startAddrDecTextBox.Text = startAddress.ToString();

                // Restore text selection
                startAddrHexTextBox.SelectionStart = selectionStart;
                startAddrHexTextBox.SelectionLength = selctionLength;

                updateStartFields = false;
            }

            if (updateEndFields)
            {
                // Save the current text selection
                int selectionStart = endAddrHexTextBox.SelectionStart;
                int selctionLength = endAddrHexTextBox.SelectionLength; 

                // Update values
                this.endAddrHexTextBox.Text = endAddress.ToString("X2");
                this.endAddrDecTextBox.Text = endAddress.ToString();
                updateEndFields = false;

                // Restore text selection
                endAddrHexTextBox.SelectionStart = selectionStart;
                endAddrHexTextBox.SelectionLength = selctionLength;
            }

            if (updateKeyFields)
            {
                // Save the current text selection
                int selectionStart = keyNumTextBox.SelectionStart;
                int selctionLength = keyNumTextBox.SelectionLength;

                // Update values
                this.keyNumTextBox.Text = keyNumber.ToString();
                updateKeyFields = false;

                // Restore text selection
                keyNumTextBox.SelectionStart = selectionStart;
                keyNumTextBox.SelectionLength = selctionLength;
            }

            // Update key type
            keyARadioButton.Checked = !keyType;
            keyBRadioButton.Checked = keyType;
        }

        private void validateValues(bool preferStartAddr)
        {
            // Ensure Start Address is 
            // - between 0 - C_LAST_BLOCK
            // - less than or equal to End Address
            if (startAddress < 0x00)
            {
                startAddress = 0x00;
                updateStartFields = true;
            }
            else if (startAddress > C_LAST_BLOCK)
            {
                startAddress = C_LAST_BLOCK;
                updateStartFields = true;
            }

            // Ensure that End Address is
            // - between 0 - C_LAST_BLOCK
            // - greater than or equal to Start Address
            if (endAddress < 0x00)
            {
                endAddress = startAddress;
                updateEndFields = true;
            }
            else if (endAddress > C_LAST_BLOCK)
            {
                endAddress = C_LAST_BLOCK;
                updateEndFields = true;
            }

            // If Start and End address are inverted then 
            // set them equal to each other
            if (startAddress > endAddress)
            {
                if (preferStartAddr)
                {
                    endAddress = startAddress;
                    updateEndFields = true;
                }
                else
                {
                    startAddress = endAddress;
                    updateStartFields = true;
                }
            }

            // Ensure the RWD Key Number is within its range
            if (keyNumber < 0x00)
            {
                keyNumber = 0x00;
            }
            else if (keyNumber > C_LAST_KEY)
            {
                keyNumber = C_LAST_KEY;
            }
        }

        private bool IsHexDigit(char _c)
        {
            if (((_c >= '0') && (_c <= '9')) || ((_c >= 'A') && (_c <= 'F')) || ((_c >= 'a') && (_c <= 'f')))
            {
                return true;
            }

            return false;
        }

        private bool IsDecDigit(char _c)
        {
            if ((_c >= '0') && (_c <= '9'))
            {
                return true;
            }

            return false;
        }

        private byte HexToByte(char _c)
        {
            byte result = 0;

            if ((_c >= '0') && (_c <= '9'))
            {
                result = (byte)(_c - '0');
            }
            else if ((_c >= 'A') && (_c <= 'F'))
            {
                result = (byte)(10 + (_c - 'A'));
            }
            else if ((_c >= 'a') && (_c <= 'f'))
            {
                result = (byte)(10 + (_c - 'a'));
            }

            return result;
        }

        private void updateHexTextKeyChars(object sender, KeyPressEventArgs e, ref byte _address, ref bool _updateFlag, bool _isStartAddr)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;

            // Ignore non hex digit key presses
            if (IsHexDigit(e.KeyChar))
            {
                // Is cursor in front of a digit                    
                // replace digit
                // move cursor to next position (select digit if one is there)
                int cur = textbox.SelectionStart;
                if (cur == 0)
                {
                    _address &= 0x0f; // clear upper byte
                    _address |= (byte)((HexToByte(e.KeyChar) << 4) & 0xf0);
                }
                else if (cur == 1)
                {
                    _address &= 0xf0; // clear lower byte
                    _address |= HexToByte(e.KeyChar);
                }

                ++cur;
                textbox.SelectionStart = Math.Min(cur, textbox.MaxLength);
                textbox.SelectionLength = (textbox.SelectionStart < 2) ? 1 : 0;
                _updateFlag = true;

                // Refresh view
                validateValues(_isStartAddr);
                refreshView();
            }
            e.Handled = true;
        }

        private void updateDecTextKeyChars(object sender, KeyPressEventArgs e, ref byte _value, ref bool _updateFlag, bool _isStartAddr)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;

            // Ignore non dec digit key presses
            if (IsDecDigit(e.KeyChar))
            {
                // Is cursor in front of a digit                    
                // replace digit
                // move cursor to next position (select digit if one is there)
                int cur = textbox.SelectionStart;
                char[] curVal = textbox.Text.ToCharArray();
                if ((cur >= 0) && (cur <=2))
                {
                    if (cur > (curVal.Length - 1))
                    {
                        char[] temp = new char[curVal.Length + 1];
                        Array.Copy(curVal, 0, temp, 0, curVal.Length);
                        temp[curVal.Length] = e.KeyChar;

                        try
                        {
                            _value = byte.Parse(new string(temp));
                        }
                        catch (OverflowException)
                        {
                            _value = byte.MaxValue;
                        }
                    }
                    else
                    {
                        curVal[cur] = e.KeyChar;
                        _value = byte.Parse(new string(curVal));
                    }

                    ++cur;
                    textbox.SelectionStart = Math.Min(cur, textbox.MaxLength);
                    textbox.SelectionLength = (textbox.SelectionStart < 2) ? 1 : 0;
                    _updateFlag = true;
                }

                // Refresh view
                validateValues(_isStartAddr);
                refreshView();
            }
            e.Handled = true;
        }

        private void updateTextboxKeyCodes(object sender, KeyEventArgs e, ref byte _value, ref bool _updateFlag, bool _isStartAddr)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;
            if (e.KeyCode == Keys.Delete)
            {
                if (textbox.SelectionLength > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues(_isStartAddr);
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
            else if (e.KeyCode == Keys.Back)
            {
                if (textbox.SelectionStart > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues(_isStartAddr);
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
        }

        #endregion

        #region Event Handlers

        private void ReadCardMemory_Load(object sender, EventArgs e)
        {
            refreshView();
        }

        private void readCardButton_Click(object sender, EventArgs e)
        {
            parent.readCard(startAddress, endAddress, keyType, keyNumber);
        }

        private void closeButton_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void startAddrNextButton_Click(object sender, EventArgs e)
        {
            // Increment the start address
            if (startAddress < C_LAST_BLOCK)
            {
                ++startAddress;
                updateStartFields = true;
            }
            validateValues(true);
            refreshView();
        }

        private void startAddrPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the start address
            if (startAddress > 0x00)
            {
                --startAddress;
                updateStartFields = true;
            }
            validateValues(true);
            refreshView();
        }

        private void endAddrNextButton_Click(object sender, EventArgs e)
        {
            // Increment the end address
            if (endAddress < C_LAST_BLOCK)
            {
                ++endAddress;
                updateEndFields = true;
            }
            validateValues(false);
            refreshView();
        }

        private void endAddrPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the end address
            if (endAddress > 0x00)
            {
                --endAddress;
                updateEndFields = true;
            }
            validateValues(false);
            refreshView();
        }

        private void keyNumNextButton_Click(object sender, EventArgs e)
        {
            // Increment the key number
            if (keyNumber < C_LAST_KEY)
            {
                ++keyNumber;
                updateKeyFields = true;
            }
            validateValues(false);
            refreshView();
        }

        private void keyNumPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the key number
            if (keyNumber > 0x00)
            {
                --keyNumber;
                updateKeyFields = true;
            }
            validateValues(false);
            refreshView();
        }

        private void startAddrHexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateHexTextKeyChars(sender, e, ref startAddress ,ref updateStartFields, true);
        }

        private void endAddrHexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateHexTextKeyChars(sender, e, ref endAddress, ref updateEndFields, false);
        }

        private void startAddrDecTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateDecTextKeyChars(sender, e, ref startAddress, ref updateStartFields, true);
        }

        private void endAddrDecTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateDecTextKeyChars(sender, e, ref endAddress, ref updateEndFields, false);
        }

        private void keyNumTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateDecTextKeyChars(sender, e, ref keyNumber, ref updateKeyFields, false);
        }
        
        private void startAddrHexTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref startAddress, ref updateStartFields, true); 
        }

        private void endAddrHexTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref endAddress, ref updateEndFields, false); 
        } 

        private void startAddrDecTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref startAddress, ref updateStartFields, true); 
        }

        private void endAddrDecTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref endAddress, ref updateEndFields, false); 
        }

        private void keyNumTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref keyNumber, ref updateKeyFields, false);
        }

        private void keyARadioButton_CheckedChanged(object sender, EventArgs e)
        {
            keyType = !keyARadioButton.Checked;
        }

        private void keyBRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            keyType = keyBRadioButton.Checked;
        }

        #endregion
    }
}
